program MailTemplate;

{
  Mail template program.
  Reads an XML configuration file,
  loads an XML template for an e-mail mail-out,
  then queries database for recipients,
  performs a mail merge and sends them out.
  Requires MSXML v3 package from Microsoft.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Version 1.0 - October 17 1999.
  Version 1.1 - April 25, 2000.
    Convert configuration file to XML.
}

{$APPTYPE CONSOLE}

uses
  Classes,
  SysUtils,
  Windows,
  MailTemplateConst in 'MailTemplateConst.pas',
  MailTemplateLog in 'MailTemplateLog.pas',
  MailTemplateQuery in 'MailTemplateQuery.pas',
  MailTemplateSender in 'MailTemplateSender.pas',
  MailTemplateMessage in 'MailTemplateMessage.pas',
  XMLConfig in 'XMLConfig.pas';

resourcestring
  EmailSent   = 'E-mail sent to: %s';
  Finished    = 'Finished MailTemplate - sent %d messages';
  LogDatabase = 'Database: ';
  LogFrom     = 'From: ';
  LogQuery    = 'Query: ';
  LogSubject  = 'Subject: ';
  LogTemplate = 'Template: ';
  Started     = 'Started MailTemplate';

const
  { Miscellaneous }
  ConfigExt    = '.xml';
  DefaultPause = 2000;

{ Open the configuration file and then load the properties }
procedure LoadMailProperties(Props: TStringList);
var
  ConfigFile: string;
begin
  if ParamCount > 0 then
    ConfigFile := ParamStr(1)
  else
    ConfigFile := ChangeFileExt(ParamStr(0), ConfigExt);
  LoadPropertiesFromXML(ConfigFile, Props);
end;

var
  Props: TStringList;

{ Return the time (ms) to pause between mailings
  so as to not overload the system }
function PauseTime: Integer;
begin
  try
    Result := StrToInt(Props.Values[PauseTimeProp]);
  except
    Result := DefaultPause;
  end;
end;

{ Are we in testing mode?
  Causes output to go to the log file rather than as e-mail }
function Testing: Boolean;
begin
  Result := (Props.Values[TestProp] = TestingValue);
end;

var
  FromEmail, ToEmail, Subject, Message: string;
  QuerySQL, EmailField: string;
  Count: Integer;
  LogFile: TMailTemplateLog;
  Template: TMailTemplateMessage;
  Query: TMailTemplateQuery;
  Sender: TMailTemplateSender;

begin
  Props    := TStringList.Create;
  LogFile  := nil;
  Template := nil;
  Query    := nil;
  Sender   := nil;
  Count    := 0;
  try
    try
      { Load the program properties }
      LoadMailProperties(Props);
      { Create and open the log file }
      LogFile    := TMailTemplateLog.Create;
      { Open the XML template document }
      Template   := TMailTemplateMessage.Create(Props.Values[TemplateProp]);
      { Extract various parameters }
      FromEmail  := Props.Values[MailFromProp];
      QuerySQL   := Template.NodeValue(QueryTag);
      Subject    := Template.NodeValue(SubjectTag);
      EmailField := Template.AttributeValue(QueryTag, EmailAttr);
      { Query the database }
      Query      := TMailTemplateQuery.Create(Props, QuerySQL);
      { Create an interface to the e-mail system }
      if not Testing then
        Sender := TMailTemplateSender.Create(Props);
      { Log parameters }
      LogFile.Log(Started);
      LogFile.Log(LogFrom + FromEmail);
      LogFile.Log(LogTemplate + Props.Values[TemplateProp]);
      LogFile.Log(LogSubject + Subject);
      LogFile.Log(LogDatabase + Props.Values[QueryAliasProp]);
      LogFile.Log(LogQuery + QuerySQL);
      { Process each record from the query }
      while not Query.EOF do
      begin
        { Get the recipient }
        ToEmail := Query.Fields.Values[EmailField];
        { Perform the mail merge - XML document with query fields }
        Message := Template.ParseMessage(Query.Fields);
        { And output the results }
        if Testing then
          logFile.LogTest(FromEmail, ToEmail, Subject, Message)
        else
        begin
          Sender.Send(FromEmail, ToEmail, Subject, Message);
          LogFile.Log(Format(EmailSent, [ToEmail]));
          { Pause so as not to overwhelm the e-mail server }
          Sleep(PauseTime);
        end;
        Inc(Count);
        Query.NextRecord;
      end;
    except on Error: Exception do
      { Catch any errors and report them }
      LogFile.Error(Error);
    end;
  finally
    LogFile.Log(Format(Finished, [Count]));
    { Tidy up }
    Props.Free;
    LogFile.Free;
    Template.Free;
    Query.Free;
    Sender.Free;
  end;
end.